function [res]=combination(expertC,ThetaR,criterium)

% Give the combination of many experts
% 
% [res]=combination(expert,constraint,n,criterium)
% 
% Inputs:
% expertC = containt the structure of the list of focal elements and
% corresponding bba for all the experts
% ThetaR = the coded and reduced discernment space
%         criterium = is the combination criterium
%                   criterium=1 Smets criterium (conjunctive rule in open world)
%                   criterium=2 Dempster-Shafer criterium (normalized) (conjunctive rule in closed world)
%                   criterium=3 Yager criterium
%                   criterium=4 disjunctive combination criterium
%                   criterium=5 Florea criterium
%                   criterium=6 PCR6 
%                   criterium=7 Mean of the bbas
%                   criterium=8 Dubois criterium (normalized and disjunctive
%                   combination)
%                   criterium=9 Dubois and Prade criterium (mixt combination)
%                   criterium=10 Mixt Combination (Martin and Osswald criterium)
%                   criterium=11 DPCR (Martin and Osswald criterium)
%                   criterium=12 MDPCR (Martin and Osswald criterium)
%                   criterium=13 Zhang's rule
%
% Output:
% res = containt the structure of the list of focal elements and
% corresponding bbas for the combinated experts
%
% March 2008 Arnaud MARTIN


switch criterium

    case 1
        %Smets criterium
        res=conjunctive(expertC);
    case 2
        %Dempster-Shafer criterium (normalized)
        expConj=conjunctive(expertC);
        ind=findeqcell(expConj.focal,[]);
        if ~isempty(ind)
            k=expConj.bba(ind);
            expConj.bba=expConj.bba/(1-k);
            expConj.bba(ind)=0;
        end
        res=expConj;
    case 3
        %Yager criterium
        expConj=conjunctive(expertC);
        ind=findeqcell(expConj.focal,[]);
        if ~isempty(ind)
            k=expConj.bba(ind);
            eTheta=ThetaR{1};
            for i=2:n
                eTheta=[union(eTheta,ThetaR{i})];
            end
            indTheta=findeqcell(expConj.focal,eTheta);
            if ~isempty(indTheta)
                expConj.bba(indTheta)=expConj.bba(indTheta)+k;
                expConj.bba(ind)=0;
            else
                sFocal=size(expConj.focal,2);
                expConj.focal(sFocal+1)={eTheta};
                expConj.bba(sFocal+1)=k;
                expConj.bba(ind)=0;
            end
        end
        res=expConj;
    case 4
        %disjounctive criterium
        [res]=disjunctive(expertC);
        
    case 5
        % Florea criterium
        expConj=conjunctive(expertC);
        expDis=disjunctive(expertC);
        
        ind=findeqcell(expConj.focal,[]);
        if ~isempty(ind)
            k=expConj.bba(ind);
            alpha=k/(1-k+k*k);
            beta=(1-k)/(1-k+k*k); 
            
            expFlo=expConj;
            expFlo.bba=beta.*expFlo.bba;
            expFlo.bba(ind)=0;    
            nbFocConj=size(expConj.focal,2);
            nbFocDis=size(expDis.focal,2);
            
            expFlo.focal(nbFocConj+1:nbFocConj+nbFocDis)=expDis.focal;
            expFlo.bba(nbFocConj+1:nbFocConj+nbFocDis)=alpha.*expDis.bba;
            
            expFlo=reduceExpert(expFlo);
        else
            expFlo=expConj;
        end
        res=expFlo;
           
    case 6
        % PCR6
        [res]=PCR6(expertC);
        
    case 7
        % Means of the bba
        [res]=meanbba(expertC);
    case 8
        % Dubois criterium (normalized and disjunctive combination)
        expDis=disjunctive(expertC);
        
        ind=findeqcell(expDis.focal,[]);
        if ~isempty(ind)
            k=expDis.bba(ind);
            expDis.bba=expDis.bba/(1-k);
            expDis.bba(ind)=0;
        end
        res=expDis;
    case 9
        % Dubois and Prade criterium (mixt combination)
        [res]=DP(expertC);
    case 10
        % Martin and Ossawald criterium (mixt combination)
        [res]=Mix(expertC);
    case 11
        % DPCR (Martin and Osswald criterium)
        [res]=DPCR(expertC);
    case 12
        % MDPCR (Martin and Osswald criterium)
        [res]=MDPCR(expertC);
    case 13
        % Zhang's rule
        [res]=Zhang(expert)

    otherwise 
        'Accident: in combination choose of criterium: uncorrect'
    
end
    
        
